import React, { useState } from "react";
import Tabs from "./Tabs";
import Editor from "react-simple-code-editor";
import { highlight, languages } from "prismjs";
import "prismjs/components/prism-json";
import "prismjs/themes/prism-tomorrow.css"; // Dark theme

export default function RequestPanel({
  url,
  method,
  body,
  setUrl,
  setMethod,
  setBody,
  onSend,
}) {
  const [activeTab, setActiveTab] = useState("body");

  const tabs = [
    { id: "params", label: "Params" },
    { id: "headers", label: "Headers" },
    { id: "body", label: "Body" },
  ];

  return (
    <div className="flex flex-col h-full">
      {/* Top Bar: Method & URL */}
      <div className="p-4 border-b border-gray-800 flex gap-2 items-center bg-gray-900">
        <select
          value={method}
          onChange={(e) => setMethod(e.target.value)}
          className="bg-gray-800 border border-gray-700 text-white text-sm rounded-lg focus:ring-orange-500 focus:border-orange-500 block p-2.5 w-28 font-bold"
        >
          <option value="GET">GET</option>
          <option value="POST">POST</option>
          <option value="PUT">PUT</option>
          <option value="DELETE">DELETE</option>
          <option value="PATCH">PATCH</option>
        </select>

        <div className="flex-1 relative">
          <input
            type="text"
            value={url}
            onChange={(e) => setUrl(e.target.value)}
            placeholder="Enter request URL"
            className="bg-gray-800 border border-gray-700 text-white text-sm rounded-lg focus:ring-orange-500 focus:border-orange-500 block w-full p-2.5 font-mono"
          />
        </div>

        <button
          onClick={onSend}
          className="text-white bg-blue-600 hover:bg-blue-700 focus:ring-4 focus:ring-blue-800 font-medium rounded-lg text-sm px-6 py-2.5"
        >
          Send
        </button>
      </div>

      {/* Tabs */}
      <Tabs tabs={tabs} activeTab={activeTab} onTabChange={setActiveTab} />

      {/* Tab Content */}
      <div className="flex-1 p-4 overflow-y-auto bg-gray-900 relative">
        {activeTab === "body" && (
          <div className="h-full flex flex-col">
            <label className="text-xs text-gray-500 mb-2">JSON Body</label>
            <div className="flex-1 border border-gray-700 rounded-lg overflow-hidden bg-gray-800 font-mono text-sm relative flex">
              {/* Line Numbers */}
              <div
                className="bg-gray-900 text-gray-500 border-r border-gray-700 text-right select-none"
                style={{
                  paddingTop: 12,
                  paddingBottom: 12,
                  fontFamily: '"Fira code", "Fira Mono", monospace',
                  fontSize: 14,
                  lineHeight: "1.5",
                  minWidth: "3rem",
                }}
              >
                {body.split("\n").map((_, i) => (
                  <div key={i} className="px-2">
                    {i + 1}
                  </div>
                ))}
              </div>

              {/* Editor */}
              <div className="flex-1 relative">
                <Editor
                  value={body}
                  onValueChange={(code) => setBody(code)}
                  highlight={(code) => highlight(code, languages.json, "json")}
                  padding={12}
                  style={{
                    fontFamily: '"Fira code", "Fira Mono", monospace',
                    fontSize: 14,
                    lineHeight: "1.5", // Strict line height to align with numbers
                    minHeight: "100%",
                  }}
                  textareaClassName="focus:outline-none"
                />
              </div>
            </div>
          </div>
        )}
        {activeTab === "params" && (
          <div className="text-gray-500 text-sm mt-4 text-center">
            Query params support coming soon.
          </div>
        )}
        {activeTab === "headers" && (
          <div className="text-gray-500 text-sm mt-4 text-center">
            Headers support coming soon.
          </div>
        )}
      </div>
    </div>
  );
}
